"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeDLocalRequest = exports.DLocalError = exports.DLOCAL_STATUS_MAPPING = exports.COUNTRY_DOCUMENT_REQUIREMENTS = exports.validateCurrency = exports.DLOCAL_SUPPORTED_CURRENCIES = exports.getDLocalHeaders = exports.verifyWebhookSignature = exports.generateSignature = exports.getDLocalConfig = void 0;
const crypto_1 = __importDefault(require("crypto"));
const error_1 = require("@b/utils/error");
// dLocal API Configuration
const getDLocalConfig = () => {
    const xLogin = process.env.APP_DLOCAL_X_LOGIN;
    const xTransKey = process.env.APP_DLOCAL_X_TRANS_KEY;
    const secretKey = process.env.APP_DLOCAL_SECRET_KEY;
    const isProduction = process.env.NODE_ENV === "production";
    if (!xLogin || !xTransKey || !secretKey) {
        throw (0, error_1.createError)({ statusCode: 500, message: "dLocal credentials are not properly configured in environment variables" });
    }
    return {
        xLogin,
        xTransKey,
        secretKey,
        baseUrl: isProduction
            ? "https://api.dlocal.com"
            : "https://sandbox.dlocal.com",
        version: "2.1",
    };
};
exports.getDLocalConfig = getDLocalConfig;
// HMAC Signature Generation for API Requests
const generateSignature = (xLogin, xDate, requestBody, secretKey) => {
    const message = xLogin + xDate + requestBody;
    const signature = crypto_1.default.createHmac("sha256", secretKey).update(message, "utf8").digest("hex");
    return `V2-HMAC-SHA256, Signature: ${signature}`;
};
exports.generateSignature = generateSignature;
// HMAC Signature Verification for Webhooks
const verifyWebhookSignature = (receivedSignature, xLogin, xDate, requestBody, secretKey) => {
    try {
        const expectedSignature = (0, exports.generateSignature)(xLogin, xDate, requestBody, secretKey);
        return receivedSignature === expectedSignature;
    }
    catch (error) {
        return false;
    }
};
exports.verifyWebhookSignature = verifyWebhookSignature;
// HTTP Headers for dLocal API
const getDLocalHeaders = (xLogin, xTransKey, xDate, signature) => ({
    "X-Date": xDate,
    "X-Login": xLogin,
    "X-Trans-Key": xTransKey,
    "Content-Type": "application/json",
    "X-Version": "2.1",
    "User-Agent": "v5-platform/1.0",
    "Authorization": signature,
});
exports.getDLocalHeaders = getDLocalHeaders;
// Currency validation for dLocal supported countries
exports.DLOCAL_SUPPORTED_CURRENCIES = [
    // Latin America
    "ARS", "BOB", "BRL", "CLP", "COP", "CRC", "DOP", "USD", "GTQ", "HNL",
    "MXN", "NIO", "PYG", "PEN", "UYU",
    // Asia
    "BDT", "CNY", "INR", "IDR", "JPY", "MYR", "PKR", "PHP", "THB", "VND",
    // Africa and Middle East
    "XAF", "CDF", "EGP", "GHS", "XOF", "JOD", "KES", "MAD", "NGN", "OMR",
    "QAR", "RWF", "SAR", "ZAR", "TZS", "TRY", "UGX", "ZMW"
];
const validateCurrency = (currency) => {
    return exports.DLOCAL_SUPPORTED_CURRENCIES.includes(currency.toUpperCase());
};
exports.validateCurrency = validateCurrency;
// Country code mapping for document requirements
exports.COUNTRY_DOCUMENT_REQUIREMENTS = {
    AR: { required: true, name: "DNI, CUIT, or CUIL", format: "Between 7 to 9 or 11 digits" },
    BD: { required: true, name: "NID Card", format: "13-17 digits" },
    BO: { required: true, name: "CI", format: "Between 5 to 20 digits" },
    BR: { required: true, name: "CPF or CNPJ", format: "11 digits for CPF, 14 for CNPJ" },
    CM: { required: true, name: "CNI or ID", format: "8 digits" },
    CL: { required: true, name: "CI or RUT", format: "Between 8 to 9 characters" },
    CN: { required: true, name: "Citizen ID", format: "18 digits or 17 digits + letter X" },
    CO: { required: true, name: "CC", format: "Between 6 to 10 digits" },
    CR: { required: true, name: "CI", format: "9 digits" },
    CD: { required: true, name: "National ID Card", format: "12 digits alphanumeric" },
    DO: { required: false, name: "ID", format: "11 digits" },
    EC: { required: true, name: "CI", format: "Between 5 to 20 digits" },
    SV: { required: true, name: "DUI", format: "9 digits" },
    EG: { required: true, name: "ID", format: "14 digits" },
    GH: { required: true, name: "Ghana Card", format: "13 digits (3 letters + 10 numbers)" },
    GT: { required: true, name: "CUI", format: "13 digits" },
    HN: { required: true, name: "DNI", format: "13 digits" },
    IN: { required: true, name: "PAN", format: "10 characters (5 letters, 4 numbers, 1 letter)" },
    ID: { required: true, name: "NIK", format: "16 digits" },
    CI: { required: true, name: "CNI", format: "11 digits" },
    JP: { required: true, name: "My Number", format: "12 digits" },
    JO: { required: true, name: "National ID Card", format: "10 digits" },
    KE: { required: true, name: "National ID Card", format: "8 digits" },
    MY: { required: true, name: "NRIC", format: "12 digits" },
    MX: { required: true, name: "CURP", format: "Between 10 to 18 characters" },
    MA: { required: true, name: "CNIE", format: "Between 5 to 20 characters" },
    NI: { required: true, name: "DNI", format: "14 digits (13 numbers, 1 letter)" },
    NE: { required: true, name: "CNI", format: "7 digits" },
    NG: { required: true, name: "NIN", format: "11 digits" },
    OM: { required: true, name: "National ID Card", format: "9 digits" },
    PK: { required: true, name: "CNIC", format: "13 digits" },
    PA: { required: false, name: "Cedula de Identidad", format: "8 digits" },
    PY: { required: true, name: "CI", format: "Between 5 to 20 digits" },
    PE: { required: true, name: "DNI", format: "Between 8 to 9 digits" },
    PH: { required: true, name: "PSN", format: "12 digits" },
    QA: { required: true, name: "National ID Card", format: "8 digits" },
    RW: { required: true, name: "National Identity Card", format: "16 digits" },
    SA: { required: true, name: "National ID Card", format: "10 digits" },
    SN: { required: true, name: "CNI or ECOWAS ID Card", format: "13 to 17 digits" },
    ZA: { required: true, name: "South African Identity Card", format: "13 digits" },
    TZ: { required: true, name: "National Identity Card", format: "20 digits" },
    TH: { required: true, name: "Thai Identity Card", format: "13 digits" },
    TR: { required: true, name: "T.C. Kimlik No.", format: "Between 5 to 20 digits" },
    UG: { required: true, name: "National ID number", format: "Between 14 to 17 digits" },
    UY: { required: true, name: "CI or RUT", format: "Between 6 to 8 digits or 12 digits" },
    VN: { required: true, name: "VNID", format: "Between 9 or 13 digits" },
    ZM: { required: false, name: "National Registration Card", format: "9 digits" },
};
// Payment status mapping
exports.DLOCAL_STATUS_MAPPING = {
    PENDING: "pending",
    PAID: "completed",
    REJECTED: "failed",
    CANCELLED: "cancelled",
    EXPIRED: "expired",
    AUTHORIZED: "authorized",
    REFUNDED: "refunded",
    PARTIALLY_REFUNDED: "partially_refunded",
    CHARGEBACK: "chargeback",
};
// Error handling
class DLocalError extends Error {
    constructor(message, code = "DLOCAL_ERROR", statusCode = 500) {
        super(message);
        this.name = "DLocalError";
        this.code = code;
        this.statusCode = statusCode;
    }
}
exports.DLocalError = DLocalError;
// API Request Helper
const makeDLocalRequest = async (endpoint, method, data) => {
    const config = (0, exports.getDLocalConfig)();
    const xDate = new Date().toISOString();
    const requestBody = data ? JSON.stringify(data) : "";
    const signature = (0, exports.generateSignature)(config.xLogin, xDate, requestBody, config.secretKey);
    const headers = (0, exports.getDLocalHeaders)(config.xLogin, config.xTransKey, xDate, signature);
    const url = `${config.baseUrl}${endpoint}`;
    const requestOptions = {
        method,
        headers,
    };
    if (data && (method === "POST" || method === "PUT")) {
        requestOptions.body = requestBody;
    }
    try {
        const response = await fetch(url, requestOptions);
        const responseData = await response.json();
        if (!response.ok) {
            throw new DLocalError(responseData.message || `dLocal API Error: ${response.status}`, responseData.code || "API_ERROR", response.status);
        }
        return responseData;
    }
    catch (error) {
        if (error instanceof DLocalError) {
            throw error;
        }
        throw new DLocalError(`Network error: ${error.message}`, "NETWORK_ERROR", 500);
    }
};
exports.makeDLocalRequest = makeDLocalRequest;
